/*
 * $Author: ac $
 * $Id: bfl106.h,v 1.1 1999/10/21 12:51:49 ac Exp $
 * $Date: 1999/10/21 12:51:49 $
 */

#ifndef __bfl106_h
#define __bfl106_h

/*----------------------------------------------------------------------*/
/*                           Nested includes                            */
/*----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>


/*----------------------------------------------------------------------*/
/*                        Constant declarations                         */
/*----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/*                          Macro declarations                          */
/*----------------------------------------------------------------------*/

#ifndef __P
#ifdef __STDC__
#define __P(args)  args
#else
#define __P(args)  ()
#endif
#endif

#ifndef NULL
#define NULL	((void*) 0)
#endif

/*----------------------------------------------------------------------*/
/*                           Bfl Memory Macros                          */
/*----------------------------------------------------------------------*/

#define bflallocate(type, size)			\
        ((type *)bflalloc((size) * sizeof(type)))
#define bflreallocate(ptr, type, size)		\
	((ptr) = (type *)bflrealloc((ptr), (size) * sizeof(type)))
#define bfldestroy(ptr)			free(ptr)

/*----------------------------------------------------------------------*/
/*                            Bfl List Macros                           */
/*----------------------------------------------------------------------*/

#define bfllistempty(list)		((list)->length == 0)
#define bfllistsize(list)		((list)->length)
#define bfllistbegin(list)		((list)->node->next)
#define bfllistend(list)		((list)->node)
#define bfllistrbegin(list)		bfllistend(list)
#define bfllistrend(list)		bfllistbegin(list)
#define bfllistnext(node)		((node)->next)
#define bfllistprev(node)		((node)->prev)
#define bfllistdata(node)		((node)->data)
#define bfllistfront(list)		bfllistdata(bfllistbegin(list))
#define bfllistback(list)		bfllistdata(bfllistprev(bfllistend(list)))
#define bfllistpushfront(list , data)	bfllistinsert(list, bfllistbegin(list), (data))
#define bfllistpushback(list, data)	bfllistinsert(list, bfllistend(list), (data))
#define bfllistpopfront(list)		bfllisterase(list, bfllistbegin(list))
#define bfllistpopback(list)		bfllisterase(list, bfllistprev(bfllistend(list)))

/*----------------------------------------------------------------------*/
/*                            Bfl Map Macros                            */
/*----------------------------------------------------------------------*/

#define bflmapleft(node)		((node)->left)
#define bflmapright(node)		((node)->right)
#define bflmapparent(node)		((node)->parent)
#define bflcolor(node)		((node)->color)
#define bflmapkey(node)			((node)->key)
#define bflmapdata(node)		((node)->data)

#define bflmaproot(map)			(bflmapparent((map)->header))
#define bflmapleftmost(map)		(bflmapleft((map)->header))
#define bflmaprightmost(map)		(bflmapright((map)->header))
#define bflmapsize(map)			((map)->size)
#define bflmapempty(map)		((map)->size == 0)
#define bflmapcompar(map)		((map)->compar)

#define bflmapbegin(map)		bflmapleft((map)->header)
#define bflmapend(map)			((map)->header)
#define bflmaprbegin(map)		bflmapright((map)->header)
#define bflmaprend(map)			((map)->header)

#define bflmapfront(map)		bflmapbegin( map )
#define bflmapback(map)			bflmaprbegin( map )

#define bflmapremove(map , key)		bflmaperase(map , bflmapfind(map , key))

/*----------------------------------------------------------------------*/
/*                            Bfl Tree Macros                           */
/*----------------------------------------------------------------------*/

#define bfltreeleft(node)		((node)->left)
#define bfltreeright(node)		((node)->right)
#define bfltreeparent(node)		((node)->parent)
#define bfltreecolor(node)		((node)->color)
#define bfltreedata(node)		((node)->data)

#define bfltreeroot(tree)		bfltreeparent((tree)->header)
#define bfltreeleftmost(tree)		bfltreeleft((tree)->header)
#define bfltreerightmost(tree)		bfltreeright((tree)->header)
#define bfltreesize(tree)		((tree)->size)
#define bfltreeempty(tree)		((tree)->size == 0)
#define bfltreecompare(tree)		((tree)->compare)

#define bfltreebegin(tree)		bfltreeleftmost(tree)
#define bfltreeend(tree)		((tree)->header)
#define bfltreerbegin(tree)		bfltreeend(tree)
#define bfltreerend(tree)		bfltreebegin(tree)

#define bfltreeremove(tree, data)	bfltreeerase(tree, bfltreefind(tree, data))

/*----------------------------------------------------------------------*/
/*                           Bfl Vector Macros                          */
/*----------------------------------------------------------------------*/

#define bflvecempty(vec)		((vec)->start == (vec)->finish)
#define bflvecsize(vec)			((vec)->finish - (vec)->start)
#define bflvecbegin(vec)		((vec)->start)
#define bflvecend(vec)			((vec)->finish)
#define bflvecrbegin(vec)		((vec)->finish - 1)
#define bflvecrend(vec)			((vec)->start - 1)
#define bflvecnext(node)		((node) + 1)
#define bflvecprev(node)		((node) - 1)
#define bflvecdata(node)		(*(node))

#define bflvecfront(vec)		bflvecbegin(vec)
#define bflvecback(vec)			bflvecend(vec)

#define bflvecpushfront(vec, data)	bflvecinsert(vec, bflvecfront(vec), data)
#define bflvecpushback(vec, data)	bflvecinsert(vec, bflvecback(vec), data)
#define bflvecpopfront(vec)		bflvecerase(vec , bflvecfront(vec))
#define bflvecpopback(vec) 		bflvecerase(vec , bflvecback(vec))
#define bflvecclean(vec)		((vec)->finish = (vec)->start)
#define bflvecnth(vec , n)		(*((vec)->start + n))
#define bflvecrand(vec)			bflvecnth(vec, bflirand(bflvecsize(vec)))
#define bflvecindex(vec, ptr)	(bflvecfindptr(vec, ptr) - (vec)->start)

/*----------------------------------------------------------------------*/
/*                            Bfl Heap Macros                           */
/*----------------------------------------------------------------------*/

#define bflheapempty(heap)		((heap)->start == (heap)->finish)
#define bflheapsize(heap)		((heap)->finish - (heap)->start)
#define bflheaptop(heap)		(*((heap)->finish - 1))

/*----------------------------------------------------------------------*/
/*                           Bfl Matrix Macros                          */
/*----------------------------------------------------------------------*/


#define bflmatpos(m, r, c)		(*((m)->data + (r) * (m)->col + (c)))
#define bflmatdata(pos)			(*(pos))
#define bflmatrowbegin(m, r)		((m)->data + (r) * (m)->col)
#define bflmatrowend(m, r)		((m)->data + (r) * (m)->col + (m)->col)
#define bflmatrownext(m, pos)		((pos) + 1)
#define bflmatcolbegin(m, c)		((m)->data + (c))
#define bflmatcolend(m, c)		((m)->data + (m)->row * (m)->col + (c))
#define bflmatcolnext(m, pos)		((pos) + (m)->col)

/*----------------------------------------------------------------------*/
/*                               Bfl Swap                               */
/*----------------------------------------------------------------------*/

#define bflswap(a, b)			(int)(a) ^= (int)(b); \
					(int)(b) ^= (int)(a); \
					(int)(a) ^= (int)(b)

/*----------------------------------------------------------------------*/
/*                          Type declarations                           */
/*----------------------------------------------------------------------*/

/* This is not a typedef so `const bflptr_t' does the right thing.  */
#ifdef __STDC__
#define bflptr_t	void *
#else
#define bflptr_t	char *
#endif

typedef unsigned int bflsize_t;

typedef int (*bflcompar_fn) __P((const bflptr_t node, const bflptr_t data));

/*----------------------------------------------------------------------*/
/*                            Bfl Chrono Type                           */
/*----------------------------------------------------------------------*/

typedef struct bfltime {
	struct timeval	 ru_utime;
	struct timeval	 ru_stime;
} bfltime_t;

typedef struct bflchrono {
	struct bfltime	 start;
	struct bfltime	 stop;
	struct bfltime	 lap;
} bflchrono_t;

/*----------------------------------------------------------------------*/
/*                             Bfl Cube Type                            */
/*----------------------------------------------------------------------*/

typedef struct bflcube {
    bflsize_t size;          /* size in bits                 */
    bflsize_t word;          /* size in words                */
    unsigned long *data;
} bflcube_t;

/*----------------------------------------------------------------------*/
/*                           Bfl Position Type                          */
/*----------------------------------------------------------------------*/

typedef struct bflpos {
    double x;
    double y;
} bflpos_t;

/*----------------------------------------------------------------------*/
/*                            Bfl List Types                            */
/*----------------------------------------------------------------------*/

typedef struct bfllistnode {
    struct bfllistnode *prev;
    struct bfllistnode *next;
    bflptr_t data;
} bfllistnode_t;

typedef struct bfllist {
    bfllistnode_t *node;
    bflsize_t length;
} bfllist_t;

/*----------------------------------------------------------------------*/
/*                             Bfl Map Types                            */
/*----------------------------------------------------------------------*/

typedef enum {
    bflblack,
    bflred
} bflcolor_t;

typedef struct bflmapnode {
    struct bflmapnode *left;	/* left child                   */
    struct bflmapnode *right;	/* right child                  */
    struct bflmapnode *parent;	/* parent                       */
    bflcolor_t color;		/* node color (black, red)      */
    bflptr_t key;		/* key of this node             */
    bflptr_t data;		/* data stored in node          */
} bflmapnode_t;

typedef struct bflmap {
    bflmapnode_t *header;
    bflsize_t size;
    bflcompar_fn compar;
    int duplicate;
} bflmap_t;

/*----------------------------------------------------------------------*/
/*                            Bfl Tree Types                            */
/*----------------------------------------------------------------------*/

typedef struct bfltreenode {
    struct bfltreenode *left;	/* left child           */
    struct bfltreenode *right;	/* right child          */
    struct bfltreenode *parent;	/* parent               */
    bflcolor_t color;		/* node color           */
    bflptr_t data;		/* data stored in node  */
} bfltreenode_t;

typedef struct bfltree {
    bfltreenode_t *header;
    bflsize_t size;
    bflcompar_fn compar;
    int duplicate;
} bfltree_t;

/*----------------------------------------------------------------------*/
/*                           Bfl Vector Types                           */
/*----------------------------------------------------------------------*/

typedef struct bflvec {
    bflptr_t *start;
    bflptr_t *finish;
    bflptr_t *end_of_storage;
} bflvec_t;

/*----------------------------------------------------------------------*/
/*                             Bfl Heap Type                            */
/*----------------------------------------------------------------------*/

typedef struct bflheap {
	bflcompar_fn	 comp;
	bflptr_t	*start;
	bflptr_t	*finish;
	bflptr_t	*end_of_storage;
} bflheap_t;

/*----------------------------------------------------------------------*/
/*                            Bfl Matrix Type                           */
/*----------------------------------------------------------------------*/

typedef struct bflmat {
	int		 row;
	int		 col;
	bflptr_t	*data;
} bflmat_t;

/*----------------------------------------------------------------------*/
/*                        Stucture declarations                         */
/*----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/*                        Variable declarations                         */
/*----------------------------------------------------------------------*/

extern char *__progname;

/*----------------------------------------------------------------------*/
/*                         Function prototypes                          */
/*----------------------------------------------------------------------*/

extern void
bflenv __P((char *progname));

/*----------------------------------------------------------------------*/
/*                         Bfl Chrono Functions                         */
/*----------------------------------------------------------------------*/

extern bflchrono_t	*bflchrononew __P(( void ));
extern void		 bflchronofree __P(( bflchrono_t *chrono ));

extern void		 bflchronoreset __P(( bflchrono_t *chrono ));
extern void		 bflchronostart __P(( bflchrono_t *chrono ));
extern void		 bflchronostop __P(( bflchrono_t *chrono ));
extern bfltime_t	 bflchronoelapsed __P(( bflchrono_t *chrono ));
extern bfltime_t	 bflchronolap __P(( bflchrono_t *chrono ));

extern char		*bflchronostring __P(( bfltime_t time ));

/*----------------------------------------------------------------------*/
/*                          Bfl Cube Functions                          */
/*----------------------------------------------------------------------*/

extern bflcube_t *bflcubenew __P((bflsize_t size));
extern bflcube_t *bflcubecopy __P((bflcube_t *cube));
extern void       bflcubefree __P((bflcube_t *cube));

extern bflcube_t *bflcubeor __P((bflcube_t *res, bflcube_t *x, bflcube_t *y));
extern bflcube_t *bflcubeand __P((bflcube_t *res, bflcube_t *x, bflcube_t *y));
extern bflcube_t *bflcubenot __P((bflcube_t *res, bflcube_t *x));

extern void       bflcubeset __P((bflcube_t *cube, bflsize_t bit));
extern void       bflcubeclr __P((bflcube_t *cube, bflsize_t bit));
extern int        bflcubebit __P((bflcube_t *cube, bflsize_t bit));

extern int        bflcubecmp __P((bflcube_t *x, bflcube_t *y));
extern void       bflcubeprint __P((FILE *fp, bflcube_t *cube));

/*----------------------------------------------------------------------*/
/*                          Bfl Error Functions                         */
/*----------------------------------------------------------------------*/

extern void
bflsyserror __P((int status, const char *fmt,...));

extern void
bflerror __P((int status, const char *fmt,...));

extern void
bflsyswarning __P((const char *fmt,...));

extern void
bflwarning __P((const char *fmt,...));

/*----------------------------------------------------------------------*/
/*                          Bfl Memory Function                         */
/*----------------------------------------------------------------------*/

extern bflptr_t
bflmemalloc __P((bflsize_t size));

extern bflptr_t
bflmemresize __P((bflptr_t ptr, bflsize_t old_size, bflsize_t new_size));

extern void
bflmemfree __P((bflptr_t ptr, bflsize_t size));

extern bflptr_t
bflalloc __P((bflsize_t size));

extern bflptr_t
bflcalloc __P((bflsize_t nmemb, bflsize_t size));

extern bflptr_t
bflrealloc __P((bflptr_t ptr, bflsize_t size));

extern void
bflfree __P((bflptr_t ptr));

/*----------------------------------------------------------------------*/
/*                          Bfl Name Functions                          */
/*----------------------------------------------------------------------*/

extern char *bflnamealloc __P((char *name));
extern void  bflnamefree __P((char *name));
extern char *bflnamefind __P((char *name));

/*----------------------------------------------------------------------*/
/*                         Bfl String Functions                         */
/*----------------------------------------------------------------------*/

extern char *
bflstrdup __P((const char *s));

extern char *
bflstrcat __P((char *str1, const char *str2));

extern char *
bflstrncat __P((char *dest, const char *src, bflsize_t n));

extern int
bflstrmatch __P((char *pattern, char *string));

/*----------------------------------------------------------------------*/
/*                         Bfl Compare Functions                        */
/*----------------------------------------------------------------------*/

extern int
bflptrcmp __P((const bflptr_t x, const bflptr_t y));

extern int
bflnumcmp __P((const bflptr_t x, const bflptr_t y));

extern int
bflstrcmp __P((const bflptr_t x, const bflptr_t y));

/*----------------------------------------------------------------------*/
/*                          Bfl Less Functions                          */
/*----------------------------------------------------------------------*/

extern int
bflptrless __P((const bflptr_t x, const bflptr_t y));

extern int
bflnumless __P((const bflptr_t x, const bflptr_t y));

extern int
bflstrless __P((const bflptr_t x, const bflptr_t y));

/*----------------------------------------------------------------------*/
/*                          Bfl List Functions                          */
/*----------------------------------------------------------------------*/

extern bfllist_t *
bfllistnew __P((void));

extern void
bfllistfree __P((bfllist_t * list));

extern bfllist_t *
bfllistcopy __P((bfllist_t * list));

extern void
bfllistclean __P((bfllist_t * list));

extern bfllistnode_t *
bfllistinsert __P((bfllist_t * list, bfllistnode_t * pos, bflptr_t data));

extern void
bfllisterase __P((bfllist_t * list, bfllistnode_t * pos));

extern bfllistnode_t *
bfllistfind __P((bfllist_t * list, bflcompar_fn compar, bflptr_t data));

extern bfllistnode_t *
bfllistfindptr __P((bfllist_t * list, bflptr_t data));

extern bfllistnode_t *
bfllistnth __P((bfllist_t * list, bflsize_t index));

extern void
bfllistremove __P((bfllist_t * list, bflcompar_fn compar, bflptr_t data));

extern void
bfllistremoveptr __P((bfllist_t * list, bflptr_t data));

extern void
bfllistreverse __P((bfllist_t * list));

extern void
bfllistmerge __P((bfllist_t * list1, bfllist_t * list2, bflcompar_fn compar));

extern void
bfllistsort __P((bfllist_t * list, bflcompar_fn compar));

/*----------------------------------------------------------------------*/
/*                           Bfl Map Functions                          */
/*----------------------------------------------------------------------*/

extern bflmap_t *
bflmapnew __P((bflcompar_fn compar, int dup));

extern void
bflmapfree __P((bflmap_t * map));

extern bflmap_t *
bflmapcopy __P((bflmap_t * map));

extern void
bflmapclean __P((bflmap_t * map));

extern bflmapnode_t *
bflmapinsert __P((bflmap_t * map, bflptr_t key, bflptr_t data));

extern void
bflmaperase __P((bflmap_t * map, bflmapnode_t * node));

extern bflmapnode_t *
bflmapfind __P((bflmap_t * map, bflptr_t key));

extern bflmapnode_t *
bflmapmin __P((bflmapnode_t * node));

extern bflmapnode_t *
bflmapmax __P((bflmapnode_t * node));

extern bflmapnode_t *
bflmapnext __P((bflmapnode_t * node));

extern bflmapnode_t *
bflmapprev __P((bflmapnode_t * node));

/*----------------------------------------------------------------------*/
/*                         Bfl Random Functions                         */
/*----------------------------------------------------------------------*/

extern int	 bflirand __P((int imax));
extern float	 bflfrand __P((void));
extern void	 bflsrand __P((unsigned int seed));

/*----------------------------------------------------------------------*/
/*                          Bfl Tree Functions                          */
/*----------------------------------------------------------------------*/

extern bfltree_t *
bfltreenew __P((bflcompar_fn compar, int dup));

extern void
bfltreefree __P((bfltree_t * tree));

extern bfltree_t *
bfltreecopy __P((bfltree_t * tree));

extern void
bfltreeclean __P((bfltree_t * tree));

extern bfltreenode_t *
bfltreeinsert __P((bfltree_t * tree, bflptr_t data));

extern void
bfltreeerase __P((bfltree_t * tree, bfltreenode_t * node));

extern bfltreenode_t *
bfltreefind __P((bfltree_t * tree, bflptr_t data));

extern bfltreenode_t *
bfltreemin __P((bfltreenode_t * x));

extern bfltreenode_t *
bfltreemax __P((bfltreenode_t * x));

extern bfltreenode_t *
bfltreenext __P((bfltreenode_t * node));

extern bfltreenode_t *
bfltreeprev __P((bfltreenode_t * node));

/*----------------------------------------------------------------------*/
/*                         Bfl Vector Functions                         */
/*----------------------------------------------------------------------*/

extern bflvec_t *
bflvecnew __P((void));

extern void
bflvecfree __P((bflvec_t * vec));

extern bflvec_t *
bflveccopy __P((bflvec_t * vec));

extern bflptr_t *
bflvecinsert __P((bflvec_t * vec, bflptr_t * pos, bflptr_t data));

extern void
bflvecerase __P((bflvec_t * vec, bflptr_t * pos));

extern bflptr_t *
bflvecfind __P((bflvec_t * vec, bflcompar_fn compar, bflptr_t data));

extern bflptr_t *
bflvecfindptr __P((bflvec_t * vec, bflptr_t data));

extern void
bflvecremove __P((bflvec_t * vec, bflcompar_fn compar, bflptr_t data));

extern void
bflvecsort __P((bflvec_t * vec, bflcompar_fn compar));

extern bflptr_t *
bflvecsearch __P((bflvec_t * vec, bflcompar_fn compar, bflptr_t data));

/*----------------------------------------------------------------------*/
/*                          Bfl Heap Functions                          */
/*----------------------------------------------------------------------*/
 
extern bflheap_t *
bflheapnew __P((bflcompar_fn comp));

extern void
bflheapfree __P((bflheap_t *heap));

extern void
bflheappush __P((bflheap_t *heap, bflptr_t data));

extern bflptr_t
bflheappop __P((bflheap_t *heap));

/*----------------------------------------------------------------------*/
/*                         Bfl Matrix Functions                         */
/*----------------------------------------------------------------------*/

extern bflmat_t *
bflmatnew __P((int row, int col));

extern void
bflmatfree __P((bflmat_t *mat));

extern void
bflmatprint __P((bflmat_t *mat, void (*print)(void *)));

/*----------------------------------------------------------------------*/
/*                          Bfl Math Functions                          */
/*----------------------------------------------------------------------*/

extern long
bfllog2 __P((long num));

extern long
bflround __P((long num));

/*----------------------------------------------------------------------*/
/*                           Bfl Sort Function                          */
/*----------------------------------------------------------------------*/

extern void
bflsort __P((bflptr_t base, bflsize_t n, bflsize_t size, bflcompar_fn compar));

/*----------------------------------------------------------------------*/
/*                        Bfl Position Functions                        */
/*----------------------------------------------------------------------*/

extern bflpos_t
bflposadd __P((bflpos_t x, bflpos_t y));

extern bflpos_t
bflpossub __P((bflpos_t x, bflpos_t y));

extern double
bflposnorm __P((bflpos_t x));

extern bflpos_t
bflposunit __P((bflpos_t x));

extern double
bflposdot __P((bflpos_t x, bflpos_t y));

#endif	/* __bfl106_h */
